package nemosofts.notes.app.receiver;

import android.Manifest;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Build;

import androidx.core.app.NotificationCompat;
import androidx.core.app.NotificationManagerCompat;
import androidx.core.content.ContextCompat;

import nemosofts.notes.app.R;
import nemosofts.notes.app.activity.CreateNoteActivity;
import nemosofts.notes.app.item.Note;
import nemosofts.notes.app.utils.helper.DBHelper;
import nemosofts.notes.app.utils.ReminderScheduler;

public class ReminderReceiver extends BroadcastReceiver {

    public static final String ACTION_NOTIFY = "nemosofts.notes.app.action.NOTIFY";
    public static final String EXTRA_NOTE_ID = "extra_note_id";

    @Override
    public void onReceive(Context context, Intent intent) {
        int noteId = intent != null ? intent.getIntExtra(EXTRA_NOTE_ID, -1) : -1;
        if (noteId <= 0) {
            return;
        }

        DBHelper dbHelper = new DBHelper(context);
        Note note = dbHelper.getNoteById(noteId);
        if (note == null) {
            return;
        }
        if (!note.isReminderEnabled()) {
            return;
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU &&
                ContextCompat.checkSelfPermission(context, Manifest.permission.POST_NOTIFICATIONS) != PackageManager.PERMISSION_GRANTED) {
            return;
        }

        ReminderScheduler.createNotificationChannel(context);

        Intent openIntent = new Intent(context, CreateNoteActivity.class);
        openIntent.putExtra("isViewOrUpdate", true);
        openIntent.putExtra("note", note);
        openIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_CLEAR_TOP);

        int flags = android.app.PendingIntent.FLAG_UPDATE_CURRENT;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            flags |= android.app.PendingIntent.FLAG_IMMUTABLE;
        }
        android.app.PendingIntent contentIntent = android.app.PendingIntent.getActivity(
                context,
                noteId,
                openIntent,
                flags
        );

        String contentTitle = note.getTitle() == null || note.getTitle().trim().isEmpty()
                ? context.getString(R.string.app_name)
                : note.getTitle();
        String contentText = note.getSubtitle() == null || note.getSubtitle().trim().isEmpty()
                ? context.getString(R.string.reminder_notification_message)
                : note.getSubtitle();

        NotificationCompat.Builder builder = new NotificationCompat.Builder(context, ReminderScheduler.CHANNEL_ID)
                .setSmallIcon(R.drawable.ic_notification)
                .setContentTitle(contentTitle)
                .setContentText(contentText)
                .setStyle(new NotificationCompat.BigTextStyle().bigText(contentText))
                .setAutoCancel(true)
                .setPriority(NotificationCompat.PRIORITY_HIGH)
                .setContentIntent(contentIntent);

        NotificationManagerCompat.from(context).notify(noteId, builder.build());
    }
}
